/*
 * Decompiled with CFR 0.152.
 */
package com.dimensionrogue.libs.mongodb.internal.connection;

import com.dimensionrogue.libs.mongodb.MongoClientException;
import com.dimensionrogue.libs.mongodb.MongoException;
import com.dimensionrogue.libs.mongodb.MongoTimeoutException;
import com.dimensionrogue.libs.mongodb.ServerAddress;
import com.dimensionrogue.libs.mongodb.annotations.ThreadSafe;
import com.dimensionrogue.libs.mongodb.assertions.Assertions;
import com.dimensionrogue.libs.mongodb.connection.ClusterConnectionMode;
import com.dimensionrogue.libs.mongodb.connection.ClusterDescription;
import com.dimensionrogue.libs.mongodb.connection.ClusterId;
import com.dimensionrogue.libs.mongodb.connection.ClusterSettings;
import com.dimensionrogue.libs.mongodb.connection.ClusterType;
import com.dimensionrogue.libs.mongodb.connection.ServerConnectionState;
import com.dimensionrogue.libs.mongodb.connection.ServerDescription;
import com.dimensionrogue.libs.mongodb.connection.ServerType;
import com.dimensionrogue.libs.mongodb.event.ClusterClosedEvent;
import com.dimensionrogue.libs.mongodb.event.ClusterDescriptionChangedEvent;
import com.dimensionrogue.libs.mongodb.event.ClusterListener;
import com.dimensionrogue.libs.mongodb.event.ClusterOpeningEvent;
import com.dimensionrogue.libs.mongodb.event.ServerDescriptionChangedEvent;
import com.dimensionrogue.libs.mongodb.internal.Locks;
import com.dimensionrogue.libs.mongodb.internal.async.SingleResultCallback;
import com.dimensionrogue.libs.mongodb.internal.connection.Cluster;
import com.dimensionrogue.libs.mongodb.internal.connection.ClusterClock;
import com.dimensionrogue.libs.mongodb.internal.connection.ClusterableServer;
import com.dimensionrogue.libs.mongodb.internal.connection.ClusterableServerFactory;
import com.dimensionrogue.libs.mongodb.internal.connection.DnsSrvRecordInitializer;
import com.dimensionrogue.libs.mongodb.internal.connection.DnsSrvRecordMonitor;
import com.dimensionrogue.libs.mongodb.internal.connection.DnsSrvRecordMonitorFactory;
import com.dimensionrogue.libs.mongodb.internal.connection.OperationContext;
import com.dimensionrogue.libs.mongodb.internal.connection.ServerTuple;
import com.dimensionrogue.libs.mongodb.internal.diagnostics.logging.Logger;
import com.dimensionrogue.libs.mongodb.internal.diagnostics.logging.Loggers;
import com.dimensionrogue.libs.mongodb.internal.event.EventListenerHelper;
import com.dimensionrogue.libs.mongodb.internal.thread.InterruptionUtil;
import com.dimensionrogue.libs.mongodb.lang.Nullable;
import com.dimensionrogue.libs.mongodb.selector.ServerSelector;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

@ThreadSafe
final class LoadBalancedCluster
implements Cluster {
    private static final Logger LOGGER = Loggers.getLogger("cluster");
    private final ClusterId clusterId;
    private final ClusterSettings settings;
    private final ClusterClock clusterClock = new ClusterClock();
    private final ClusterListener clusterListener;
    private ClusterDescription description;
    @Nullable
    private ClusterableServer server;
    private final AtomicBoolean closed = new AtomicBoolean();
    private final DnsSrvRecordMonitor dnsSrvRecordMonitor;
    private volatile MongoException srvResolutionException;
    private boolean srvRecordResolvedToMultipleHosts;
    private volatile boolean initializationCompleted;
    private List<ServerSelectionRequest> waitQueue = new LinkedList<ServerSelectionRequest>();
    private Thread waitQueueHandler;
    private final Lock lock = new ReentrantLock(true);
    private final Condition condition = this.lock.newCondition();

    LoadBalancedCluster(final ClusterId clusterId, ClusterSettings settings, final ClusterableServerFactory serverFactory, DnsSrvRecordMonitorFactory dnsSrvRecordMonitorFactory) {
        Assertions.assertTrue(settings.getMode() == ClusterConnectionMode.LOAD_BALANCED);
        LOGGER.info(String.format("Cluster created with id %s and settings %s", clusterId, settings.getShortDescription()));
        this.clusterId = clusterId;
        this.settings = settings;
        this.clusterListener = EventListenerHelper.singleClusterListener(settings);
        this.description = new ClusterDescription(settings.getMode(), ClusterType.UNKNOWN, Collections.emptyList(), settings, serverFactory.getSettings());
        if (settings.getSrvHost() == null) {
            this.dnsSrvRecordMonitor = null;
            this.init(clusterId, serverFactory, settings.getHosts().get(0));
            this.initializationCompleted = true;
        } else {
            Assertions.notNull("dnsSrvRecordMonitorFactory", dnsSrvRecordMonitorFactory);
            this.dnsSrvRecordMonitor = dnsSrvRecordMonitorFactory.create(Assertions.assertNotNull(settings.getSrvHost()), settings.getSrvServiceName(), new DnsSrvRecordInitializer(){

                @Override
                public void initialize(Collection<ServerAddress> hosts) {
                    List localWaitQueue;
                    LOGGER.info("SRV resolution completed with hosts: " + hosts);
                    LoadBalancedCluster.this.lock.lock();
                    try {
                        if (LoadBalancedCluster.this.isClosed()) {
                            return;
                        }
                        LoadBalancedCluster.this.srvResolutionException = null;
                        if (hosts.size() != 1) {
                            LoadBalancedCluster.this.srvRecordResolvedToMultipleHosts = true;
                        } else {
                            LoadBalancedCluster.this.init(clusterId, serverFactory, hosts.iterator().next());
                        }
                        LoadBalancedCluster.this.initializationCompleted = true;
                        localWaitQueue = LoadBalancedCluster.this.waitQueue;
                        LoadBalancedCluster.this.waitQueue = Collections.emptyList();
                        LoadBalancedCluster.this.condition.signalAll();
                    }
                    finally {
                        LoadBalancedCluster.this.lock.unlock();
                    }
                    localWaitQueue.forEach(request -> LoadBalancedCluster.this.handleServerSelectionRequest(request));
                }

                @Override
                public void initialize(MongoException initializationException) {
                    LoadBalancedCluster.this.srvResolutionException = initializationException;
                }

                @Override
                public ClusterType getClusterType() {
                    return LoadBalancedCluster.this.initializationCompleted ? ClusterType.LOAD_BALANCED : ClusterType.UNKNOWN;
                }
            });
            this.dnsSrvRecordMonitor.start();
        }
    }

    private void init(ClusterId clusterId, ClusterableServerFactory serverFactory, ServerAddress host) {
        this.clusterListener.clusterOpening(new ClusterOpeningEvent(clusterId));
        ClusterDescription initialDescription = new ClusterDescription(this.settings.getMode(), ClusterType.LOAD_BALANCED, Collections.singletonList(ServerDescription.builder().address(this.settings.getHosts().get(0)).state(ServerConnectionState.CONNECTING).build()), this.settings, serverFactory.getSettings());
        this.clusterListener.clusterDescriptionChanged(new ClusterDescriptionChangedEvent(clusterId, initialDescription, this.description));
        this.description = new ClusterDescription(ClusterConnectionMode.LOAD_BALANCED, ClusterType.LOAD_BALANCED, Collections.singletonList(ServerDescription.builder().ok(true).state(ServerConnectionState.CONNECTED).type(ServerType.LOAD_BALANCER).address(host).build()), this.settings, serverFactory.getSettings());
        this.server = serverFactory.create(this, host);
        this.clusterListener.clusterDescriptionChanged(new ClusterDescriptionChangedEvent(clusterId, this.description, initialDescription));
    }

    @Override
    public ClusterSettings getSettings() {
        Assertions.isTrue("open", !this.isClosed());
        return this.settings;
    }

    @Override
    public ClusterId getClusterId() {
        return this.clusterId;
    }

    @Override
    public ClusterableServer getServer(ServerAddress serverAddress) {
        Assertions.isTrue("open", !this.isClosed());
        this.waitForSrv();
        return Assertions.assertNotNull(this.server);
    }

    @Override
    public ClusterDescription getCurrentDescription() {
        Assertions.isTrue("open", !this.isClosed());
        return this.description;
    }

    @Override
    public ClusterClock getClock() {
        Assertions.isTrue("open", !this.isClosed());
        return this.clusterClock;
    }

    @Override
    public ServerTuple selectServer(ServerSelector serverSelector, OperationContext operationContext) {
        Assertions.isTrue("open", !this.isClosed());
        this.waitForSrv();
        if (this.srvRecordResolvedToMultipleHosts) {
            throw this.createResolvedToMultipleHostsException();
        }
        return new ServerTuple(Assertions.assertNotNull(this.server), this.description.getServerDescriptions().get(0));
    }

    private void waitForSrv() {
        if (this.initializationCompleted) {
            return;
        }
        Locks.withLock(this.lock, () -> {
            long remainingTimeNanos = this.getMaxWaitTimeNanos();
            while (!this.initializationCompleted) {
                if (this.isClosed()) {
                    throw this.createShutdownException();
                }
                if (remainingTimeNanos <= 0L) {
                    throw this.createTimeoutException();
                }
                try {
                    remainingTimeNanos = this.condition.awaitNanos(remainingTimeNanos);
                }
                catch (InterruptedException e) {
                    throw InterruptionUtil.interruptAndCreateMongoInterruptedException(String.format("Interrupted while resolving SRV records for %s", this.settings.getSrvHost()), e);
                }
            }
        });
    }

    @Override
    public void selectServerAsync(ServerSelector serverSelector, OperationContext operationContext, SingleResultCallback<ServerTuple> callback) {
        if (this.isClosed()) {
            callback.onResult(null, this.createShutdownException());
            return;
        }
        ServerSelectionRequest serverSelectionRequest = new ServerSelectionRequest(this.getMaxWaitTimeNanos(), callback);
        if (this.initializationCompleted) {
            this.handleServerSelectionRequest(serverSelectionRequest);
        } else {
            this.notifyWaitQueueHandler(serverSelectionRequest);
        }
    }

    private MongoClientException createShutdownException() {
        return new MongoClientException("Shutdown in progress");
    }

    @Override
    public void close() {
        if (!this.closed.getAndSet(true)) {
            ClusterableServer localServer;
            LOGGER.info(String.format("Cluster closed with id %s", this.clusterId));
            if (this.dnsSrvRecordMonitor != null) {
                this.dnsSrvRecordMonitor.close();
            }
            if ((localServer = Locks.withLock(this.lock, () -> {
                this.condition.signalAll();
                return this.server;
            })) != null) {
                localServer.close();
            }
            this.clusterListener.clusterClosed(new ClusterClosedEvent(this.clusterId));
        }
    }

    @Override
    public boolean isClosed() {
        return this.closed.get();
    }

    @Override
    public void withLock(Runnable action) {
        Assertions.fail();
    }

    @Override
    public void onChange(ServerDescriptionChangedEvent event) {
        Assertions.fail();
    }

    private void handleServerSelectionRequest(ServerSelectionRequest serverSelectionRequest) {
        Assertions.assertTrue(this.initializationCompleted);
        if (this.srvRecordResolvedToMultipleHosts) {
            serverSelectionRequest.onError(this.createResolvedToMultipleHostsException());
        } else {
            serverSelectionRequest.onSuccess(new ServerTuple(Assertions.assertNotNull(this.server), this.description.getServerDescriptions().get(0)));
        }
    }

    private MongoClientException createResolvedToMultipleHostsException() {
        return new MongoClientException("In load balancing mode, the host must resolve to a single SRV record, but instead it resolved to multiple hosts");
    }

    private MongoTimeoutException createTimeoutException() {
        MongoException localSrvResolutionException = this.srvResolutionException;
        if (localSrvResolutionException == null) {
            return new MongoTimeoutException(String.format("Timed out after %d ms while waiting to resolve SRV records for %s.", this.settings.getServerSelectionTimeout(TimeUnit.MILLISECONDS), this.settings.getSrvHost()));
        }
        return new MongoTimeoutException(String.format("Timed out after %d ms while waiting to resolve SRV records for %s. Resolution exception was '%s'", this.settings.getServerSelectionTimeout(TimeUnit.MILLISECONDS), this.settings.getSrvHost(), localSrvResolutionException));
    }

    private long getMaxWaitTimeNanos() {
        if (this.settings.getServerSelectionTimeout(TimeUnit.NANOSECONDS) < 0L) {
            return Long.MAX_VALUE;
        }
        return this.settings.getServerSelectionTimeout(TimeUnit.NANOSECONDS);
    }

    private void notifyWaitQueueHandler(ServerSelectionRequest request) {
        Locks.withLock(this.lock, () -> {
            if (this.isClosed()) {
                request.onError(this.createShutdownException());
                return;
            }
            if (this.initializationCompleted) {
                this.handleServerSelectionRequest(request);
                return;
            }
            this.waitQueue.add(request);
            if (this.waitQueueHandler == null) {
                this.waitQueueHandler = new Thread((Runnable)new WaitQueueHandler(), "cluster-" + this.clusterId.getValue());
                this.waitQueueHandler.setDaemon(true);
                this.waitQueueHandler.start();
            } else {
                this.condition.signalAll();
            }
        });
    }

    private static final class ServerSelectionRequest {
        private final long maxWaitTimeNanos;
        private final long startTimeNanos = System.nanoTime();
        private final SingleResultCallback<ServerTuple> callback;

        private ServerSelectionRequest(long maxWaitTimeNanos, SingleResultCallback<ServerTuple> callback) {
            this.maxWaitTimeNanos = maxWaitTimeNanos;
            this.callback = callback;
        }

        long getRemainingTime(long curTimeNanos) {
            return this.startTimeNanos + this.maxWaitTimeNanos - curTimeNanos;
        }

        public void onSuccess(ServerTuple serverTuple) {
            try {
                this.callback.onResult(serverTuple, null);
            }
            catch (Exception e) {
                LOGGER.warn("Unanticipated exception thrown from callback", e);
            }
        }

        public void onError(Throwable exception) {
            try {
                this.callback.onResult(null, exception);
            }
            catch (Exception e) {
                LOGGER.warn("Unanticipated exception thrown from callback", e);
            }
        }
    }

    private final class WaitQueueHandler
    implements Runnable {
        private WaitQueueHandler() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            ArrayList<ServerSelectionRequest> timeoutList = new ArrayList<ServerSelectionRequest>();
            while (!LoadBalancedCluster.this.isClosed() && !LoadBalancedCluster.this.initializationCompleted) {
                LoadBalancedCluster.this.lock.lock();
                try {
                    if (LoadBalancedCluster.this.isClosed() || LoadBalancedCluster.this.initializationCompleted) break;
                    long waitTimeNanos = Long.MAX_VALUE;
                    long curTimeNanos = System.nanoTime();
                    Iterator iterator = LoadBalancedCluster.this.waitQueue.iterator();
                    while (iterator.hasNext()) {
                        ServerSelectionRequest next = (ServerSelectionRequest)iterator.next();
                        long remainingTime = next.getRemainingTime(curTimeNanos);
                        if (remainingTime <= 0L) {
                            timeoutList.add(next);
                            iterator.remove();
                            continue;
                        }
                        waitTimeNanos = Math.min(remainingTime, waitTimeNanos);
                    }
                    if (timeoutList.isEmpty()) {
                        try {
                            LoadBalancedCluster.this.condition.await(waitTimeNanos, TimeUnit.NANOSECONDS);
                        }
                        catch (InterruptedException unexpected) {
                            Assertions.fail();
                        }
                    }
                }
                finally {
                    LoadBalancedCluster.this.lock.unlock();
                }
                timeoutList.forEach(request -> request.onError(LoadBalancedCluster.this.createTimeoutException()));
                timeoutList.clear();
            }
            List shutdownList = Locks.withLock(LoadBalancedCluster.this.lock, () -> {
                ArrayList result = new ArrayList(LoadBalancedCluster.this.waitQueue);
                LoadBalancedCluster.this.waitQueue.clear();
                return result;
            });
            shutdownList.forEach(request -> request.onError(LoadBalancedCluster.this.createShutdownException()));
        }
    }
}

