/*
 * Decompiled with CFR 0.152.
 */
package com.dimensionrogue.database;

import com.dimensionrogue.DimensionRogue;
import com.dimensionrogue.libs.bson.Document;
import com.dimensionrogue.libs.mongodb.client.MongoClient;
import com.dimensionrogue.libs.mongodb.client.MongoClients;
import com.dimensionrogue.libs.mongodb.client.MongoCollection;
import com.dimensionrogue.libs.mongodb.client.MongoDatabase;
import com.dimensionrogue.libs.mongodb.client.model.ReplaceOptions;
import com.dimensionrogue.models.DimensionData;
import com.dimensionrogue.models.PlayerData;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Reader;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;

public class DatabaseManager {
    private final DimensionRogue plugin;
    private final DatabaseType type;
    private final Gson gson;
    private MongoClient mongoClient;
    private MongoDatabase mongoDatabase;
    private Path dataFolder;

    public DatabaseManager(DimensionRogue plugin) {
        this.plugin = plugin;
        this.gson = new GsonBuilder().setPrettyPrinting().create();
        String typeStr = plugin.getConfig().getString("database.type", "FLATFILE");
        this.type = DatabaseType.valueOf(typeStr.toUpperCase());
        switch (this.type.ordinal()) {
            case 1: {
                this.initMongoDB();
                break;
            }
            case 2: {
                this.initMySQL();
                break;
            }
            default: {
                this.initFlatfile();
            }
        }
    }

    private void initMongoDB() {
        try {
            String uri = this.plugin.getConfig().getString("database.mongodb.uri");
            String dbName = this.plugin.getConfig().getString("database.mongodb.database");
            this.mongoClient = MongoClients.create(uri);
            this.mongoDatabase = this.mongoClient.getDatabase(dbName);
            this.plugin.getLogger().info("MongoDB connected successfully!");
        }
        catch (Exception e) {
            this.plugin.getLogger().severe("Failed to connect to MongoDB: " + e.getMessage());
            this.plugin.getLogger().info("Falling back to FLATFILE storage");
            this.initFlatfile();
        }
    }

    private void initMySQL() {
        this.plugin.getLogger().warning("MySQL not yet implemented, using FLATFILE");
        this.initFlatfile();
    }

    private void initFlatfile() {
        this.dataFolder = Paths.get(this.plugin.getDataFolder().getAbsolutePath(), "data");
        try {
            Files.createDirectories(this.dataFolder, new FileAttribute[0]);
            Files.createDirectories(this.dataFolder.resolve("players"), new FileAttribute[0]);
            Files.createDirectories(this.dataFolder.resolve("dimensions"), new FileAttribute[0]);
            Files.createDirectories(this.dataFolder.resolve("parties"), new FileAttribute[0]);
            Files.createDirectories(this.dataFolder.resolve("leaderboards"), new FileAttribute[0]);
            this.plugin.getLogger().info("Flatfile storage initialized!");
        }
        catch (IOException e) {
            this.plugin.getLogger().severe("Failed to create data directories: " + e.getMessage());
        }
    }

    public CompletableFuture<PlayerData> loadPlayerData(UUID uuid) {
        return CompletableFuture.supplyAsync(() -> {
            if (this.type == DatabaseType.FLATFILE) {
                return this.loadPlayerDataFlatfile(uuid);
            }
            if (this.type == DatabaseType.MONGODB) {
                return this.loadPlayerDataMongo(uuid);
            }
            return new PlayerData(uuid);
        });
    }

    public void savePlayerData(PlayerData data) {
        CompletableFuture.runAsync(() -> {
            if (this.type == DatabaseType.FLATFILE) {
                this.savePlayerDataFlatfile(data);
            } else if (this.type == DatabaseType.MONGODB) {
                this.savePlayerDataMongo(data);
            }
        });
    }

    private PlayerData loadPlayerDataFlatfile(UUID uuid) {
        PlayerData playerData;
        Path file = this.dataFolder.resolve("players").resolve(uuid.toString() + ".json");
        if (!Files.exists(file, new LinkOption[0])) {
            return new PlayerData(uuid);
        }
        FileReader reader = new FileReader(file.toFile());
        try {
            PlayerData data = (PlayerData)this.gson.fromJson((Reader)reader, PlayerData.class);
            playerData = data != null ? data : new PlayerData(uuid);
        }
        catch (Throwable throwable) {
            try {
                try {
                    ((Reader)reader).close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                this.plugin.getLogger().warning("Failed to load player data for " + String.valueOf(uuid));
                return new PlayerData(uuid);
            }
        }
        ((Reader)reader).close();
        return playerData;
    }

    private void savePlayerDataFlatfile(PlayerData data) {
        Path file = this.dataFolder.resolve("players").resolve(data.getUuid().toString() + ".json");
        try (FileWriter writer = new FileWriter(file.toFile());){
            this.gson.toJson((Object)data, (Appendable)writer);
        }
        catch (IOException e) {
            this.plugin.getLogger().warning("Failed to save player data for " + String.valueOf(data.getUuid()));
        }
    }

    private PlayerData loadPlayerDataMongo(UUID uuid) {
        try {
            MongoCollection<Document> collection = this.mongoDatabase.getCollection("players");
            Document doc = (Document)collection.find(new Document("uuid", uuid.toString())).first();
            if (doc == null) {
                return new PlayerData(uuid);
            }
            String json = doc.toJson();
            PlayerData data = (PlayerData)this.gson.fromJson(json, PlayerData.class);
            return data != null ? data : new PlayerData(uuid);
        }
        catch (Exception e) {
            this.plugin.getLogger().warning("Failed to load player data from MongoDB: " + e.getMessage());
            return new PlayerData(uuid);
        }
    }

    private void savePlayerDataMongo(PlayerData data) {
        try {
            MongoCollection<Document> collection = this.mongoDatabase.getCollection("players");
            String json = this.gson.toJson((Object)data);
            Document doc = Document.parse(json);
            doc.put("uuid", (Object)data.getUuid().toString());
            collection.replaceOne(new Document("uuid", data.getUuid().toString()), doc, new ReplaceOptions().upsert(true));
        }
        catch (Exception e) {
            this.plugin.getLogger().warning("Failed to save player data to MongoDB: " + e.getMessage());
        }
    }

    public CompletableFuture<DimensionData> loadDimensionData(UUID ownerId) {
        return CompletableFuture.supplyAsync(() -> {
            if (this.type == DatabaseType.FLATFILE) {
                return this.loadDimensionDataFlatfile(ownerId);
            }
            if (this.type == DatabaseType.MONGODB) {
                return this.loadDimensionDataMongo(ownerId);
            }
            return new DimensionData(ownerId);
        });
    }

    public void saveDimensionData(DimensionData data) {
        CompletableFuture.runAsync(() -> {
            if (this.type == DatabaseType.FLATFILE) {
                this.saveDimensionDataFlatfile(data);
            } else if (this.type == DatabaseType.MONGODB) {
                this.saveDimensionDataMongo(data);
            }
        });
    }

    private DimensionData loadDimensionDataFlatfile(UUID ownerId) {
        DimensionData dimensionData;
        Path file = this.dataFolder.resolve("dimensions").resolve(ownerId.toString() + ".json");
        if (!Files.exists(file, new LinkOption[0])) {
            return new DimensionData(ownerId);
        }
        FileReader reader = new FileReader(file.toFile());
        try {
            DimensionData data = (DimensionData)this.gson.fromJson((Reader)reader, DimensionData.class);
            dimensionData = data != null ? data : new DimensionData(ownerId);
        }
        catch (Throwable throwable) {
            try {
                try {
                    ((Reader)reader).close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                this.plugin.getLogger().warning("Failed to load dimension data for " + String.valueOf(ownerId));
                return new DimensionData(ownerId);
            }
        }
        ((Reader)reader).close();
        return dimensionData;
    }

    private void saveDimensionDataFlatfile(DimensionData data) {
        Path file = this.dataFolder.resolve("dimensions").resolve(data.getOwnerId().toString() + ".json");
        try (FileWriter writer = new FileWriter(file.toFile());){
            this.gson.toJson((Object)data, (Appendable)writer);
        }
        catch (IOException e) {
            this.plugin.getLogger().warning("Failed to save dimension data for " + String.valueOf(data.getOwnerId()));
        }
    }

    private DimensionData loadDimensionDataMongo(UUID ownerId) {
        try {
            MongoCollection<Document> collection = this.mongoDatabase.getCollection("dimensions");
            Document doc = (Document)collection.find(new Document("ownerId", ownerId.toString())).first();
            if (doc == null) {
                return new DimensionData(ownerId);
            }
            String json = doc.toJson();
            DimensionData data = (DimensionData)this.gson.fromJson(json, DimensionData.class);
            return data != null ? data : new DimensionData(ownerId);
        }
        catch (Exception e) {
            this.plugin.getLogger().warning("Failed to load dimension data from MongoDB: " + e.getMessage());
            return new DimensionData(ownerId);
        }
    }

    private void saveDimensionDataMongo(DimensionData data) {
        try {
            MongoCollection<Document> collection = this.mongoDatabase.getCollection("dimensions");
            String json = this.gson.toJson((Object)data);
            Document doc = Document.parse(json);
            doc.put("ownerId", (Object)data.getOwnerId().toString());
            collection.replaceOne(new Document("ownerId", data.getOwnerId().toString()), doc, new ReplaceOptions().upsert(true));
        }
        catch (Exception e) {
            this.plugin.getLogger().warning("Failed to save dimension data to MongoDB: " + e.getMessage());
        }
    }

    public void close() {
        if (this.mongoClient != null) {
            this.mongoClient.close();
        }
    }

    public static enum DatabaseType {
        FLATFILE,
        MONGODB,
        MYSQL;

    }
}

